#!/usr/bin/env python3
"""Sweep coupling and exponent parameters for the realistic kernel.

This driver script loads a NumPy kernel built by
``kernel_builder/make_kernel_from_eigs.py`` and sweeps over the coupling
``b`` and exponent ``k`` arrays defined in the main section.  For each
parameter pair it computes the fundamental string tension,
rescales it to the adjoint representation using Casimir scaling, and
records the results in a CSV file under ``results/adjoint_ratio.csv``.

To run the sweep after generating ``kernel.npy``:

    python scripts/run_adjoint_real_kernel.py

Make sure to install dependencies with ``pip install -r requirements.txt``
and generate the kernel via ``kernel_builder/make_kernel_from_eigs.py``
before executing this script.
"""

import os
import sys
import numpy as np
import pandas as pd

# Make sure the parent directory (repository root) is on the module search
# path so that 'simulation' can be imported when this script is
# executed directly.  Without this, Python will not find the local
# package when running ``python scripts/run_adjoint_real_kernel.py``.
repo_root = os.path.abspath(os.path.join(os.path.dirname(__file__), '..'))
if repo_root not in sys.path:
    sys.path.insert(0, repo_root)

from simulation.compute_tension import fundamental_string_tension, adjoint_string_tension


def main() -> None:
    # Coupling values and kernel exponents to sweep.  These arrays
    # reflect the user’s specification from the assignment.
    b_values = [2.5, 3.0, 3.25, 3.5, 4.0, 4.25]
    k_values = [0.1, 0.25, 0.5, 0.75, 1.0]

    # Load the pre‑computed kernel.  If the file does not exist,
    # instruct the user to run the kernel builder first.
    kernel_path = os.path.join('kernel_builder', 'kernel.npy')
    if not os.path.exists(kernel_path):
        raise FileNotFoundError(
            f"Kernel file '{kernel_path}' not found. Please run "
            f"'python kernel_builder/make_kernel_from_eigs.py --config config.yaml --input_csv "
            f"kernel_builder/kernel_eigs.csv --output {kernel_path}' first."
        )
    kernel = np.load(kernel_path)

    results = []
    for b in b_values:
        for k in k_values:
            sigma_fund = fundamental_string_tension(kernel, b, k)
            sigma_adj = adjoint_string_tension(sigma_fund, N=3)
            R = sigma_adj / sigma_fund if sigma_fund != 0 else float('inf')
            results.append({
                'b': b,
                'k': k,
                'σ_fund': sigma_fund,
                'σ_adj': sigma_adj,
                'R': R
            })

    df = pd.DataFrame(results)
    # Ensure the results directory exists
    os.makedirs('results', exist_ok=True)
    output_csv = os.path.join('results', 'adjoint_ratio.csv')
    df.to_csv(output_csv, index=False)
    print(f"Wrote results to {output_csv}")


if __name__ == '__main__':
    main()
